// Script to create a task with past periods for testing
const { PrismaClient } = require("@prisma/client");
const prisma = new PrismaClient();

async function createTaskWithPastPeriods() {
  console.log("🌱 Creating task with past periods...");

  try {
    // Get admin and technician
    const admin = await prisma.user.findFirst({
      where: { role: "ADMIN" },
    });

    const technician = await prisma.user.findFirst({
      where: { role: "TECHNICIAN" },
    });

    if (!admin || !technician) {
      console.error("❌ Need admin and technician users");
      return;
    }

    // Get a department and location
    const department = await prisma.department.findFirst();
    const location = await prisma.location.findFirst({
      where: { departmentId: department?.id },
    });

    if (!department || !location) {
      console.error("❌ Need department and location");
      return;
    }

    // Calculate dates
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    // Start date: 30 days ago
    const startDate = new Date(today);
    startDate.setDate(startDate.getDate() - 30);
    startDate.setHours(0, 0, 0, 0);

    console.log(`Start Date: ${startDate.toLocaleDateString()}`);
    console.log(`Today: ${today.toLocaleDateString()}`);
    console.log(
      `Days since start: ${Math.floor(
        (today - startDate) / (1000 * 60 * 60 * 24)
      )}`
    );

    // Create task
    const task = await prisma.task.create({
      data: {
        title: "Daily Equipment Check - Test with Past Periods",
        description:
          "This task has a start date 30 days ago, so you should see 30 past periods in the history.",
        isRecurring: true,
        recurrencePattern: "daily",
        startDate: startDate,
        assignedToId: technician.id,
        createdById: admin.id,
        scopeType: "location",
        locationId: location.id,
        departmentId: department.id,
        attachmentsJson: "[]",
        selectedLocationsJson: null,
        selectedElementsJson: null,
      },
    });

    console.log(`✅ Created task: ${task.title}`);
    console.log(`   ID: ${task.id}`);
    console.log(`   Start Date: ${task.startDate?.toLocaleDateString()}`);

    // Create period completions for past 20 days (some completed, some not)
    console.log("\n📅 Creating period completions...");
    let completedCount = 0;
    let skippedCount = 0;

    for (let i = 20; i >= 1; i--) {
      const periodDate = new Date(today);
      periodDate.setDate(periodDate.getDate() - i);
      periodDate.setHours(0, 0, 0, 0);

      // Complete about 70% of past periods
      if (i % 3 !== 0) {
        try {
          await prisma.taskPeriodCompletion.create({
            data: {
              taskId: task.id,
              periodDate: periodDate,
              completedAt: new Date(periodDate.getTime() + 10 * 60 * 60 * 1000), // 10 AM
              completedById: technician.id,
            },
          });
          completedCount++;
          if (i <= 5) {
            console.log(
              `  ✅ Completed period: ${periodDate.toLocaleDateString()}`
            );
          }
        } catch (error) {
          // Period might already exist
        }
      } else {
        skippedCount++;
        if (i <= 5) {
          console.log(
            `  ⏭️  Skipped period (not completed): ${periodDate.toLocaleDateString()}`
          );
        }
      }
    }

    console.log(`\n✨ Summary:`);
    console.log(`   - Task ID: ${task.id}`);
    console.log(
      `   - Start Date: ${startDate.toLocaleDateString()} (30 days ago)`
    );
    console.log(`   - Completed periods: ${completedCount}`);
    console.log(`   - Incomplete periods: ${skippedCount}`);
    console.log(`   - Total past periods: ${completedCount + skippedCount}`);
    console.log(
      `\n💡 Open this task and click 'History' to see all past periods!`
    );
  } catch (error) {
    console.error("❌ Error:", error);
    throw error;
  }
}

createTaskWithPastPeriods()
  .catch((e) => {
    console.error("❌ Failed:", e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
